import cv2
import tkinter as tk
from PIL import Image, ImageTk
import mediapipe as mp
import numpy as np
import pandas as pd
from predict import angle
from itertools import combinations
import threading
import time
import queue
import numpy as np

# remove warnings
import warnings
warnings.filterwarnings("ignore")

# make a frame queue
frame_queue = queue.Queue(10)

# GUI class
class PoseEstimationApp:
    def __init__(self, window, window_title):
        self.window = window
        self.window.title(window_title)
        
        self.cap = cv2.VideoCapture(0)  # Live camera feed
        
        self.mp_pose = mp.solutions.pose
        self.mp_draw = mp.solutions.drawing_utils
        self.pose = self.mp_pose.Pose()
        
        self.canvas_camera = tk.Canvas(window)
        self.canvas_camera.pack(expand=True, fill=tk.BOTH)
        
        self.canvas_pose = tk.Canvas(window, width=300, height=225)
        self.canvas_pose.place(relx=.97, rely=0.95, anchor=tk.SE)
        
        self.update()
        self.window.mainloop()
    
    def process_frame(self, frame):
        global frame_queue

        results = self.pose.process(frame)     
        if results.pose_landmarks:
            data = []
            for landmark in results.pose_landmarks.landmark:
                    data += [landmark.x, landmark.y, landmark.z]

            if frame_queue.qsize() <= 10:
                frame_queue.put(data)
            else:
                print("queue full")
                frame_queue.get()
                frame_queue.put(data)

            preprocess()

        return results

    def update(self):
        ret, frame = self.cap.read()
        if ret:
            img = cv2.resize(frame, (self.window.winfo_width(), self.window.winfo_height()))

            # Move the results assignment inside the if block
            results = self.process_frame(img)
            
            # Draw pose landmarks on the live camera feed
            self.mp_draw.draw_landmarks(img, results.pose_landmarks, self.mp_pose.POSE_CONNECTIONS,
                                        self.mp_draw.DrawingSpec((255, 0, 0), 2, 2),
                                        self.mp_draw.DrawingSpec((255, 0, 255), 2, 2)
                                        )
            
            photo_img = ImageTk.PhotoImage(image=Image.fromarray(cv2.cvtColor(img, cv2.COLOR_BGR2RGB)))
            self.canvas_camera.create_image(0, 0, image=photo_img, anchor=tk.NW)
            self.canvas_camera.photo = photo_img
            
            # Extract the pose landmarks and display in the PiP canvas
            pose_img = np.zeros((self.canvas_pose.winfo_height(), self.canvas_pose.winfo_width(), 3), dtype=np.uint8)
            pose_img.fill(0)
            self.mp_draw.draw_landmarks(pose_img, results.pose_landmarks, self.mp_pose.POSE_CONNECTIONS,
                                        self.mp_draw.DrawingSpec((255, 0, 0), 2, 2),
                                        self.mp_draw.DrawingSpec((255, 0, 255), 2, 2)
                                        )
            pose_photo = ImageTk.PhotoImage(image=Image.fromarray(cv2.cvtColor(pose_img, cv2.COLOR_BGR2RGB)))
            self.canvas_pose.create_image(0, 0, image=pose_photo, anchor=tk.NW)
            self.canvas_pose.photo = pose_photo
        
        self.window.after(10, self.update)

    
    def __del__(self):
        if self.cap.isOpened():
            self.cap.release()

# run this in a seperate thread 
def preprocess():
    # calculate the std between the first 10 frames
    global frame_queue

    # find std of the first and 10th frame
    threshold = 0.1
    std = []
    print(frame_queue.qsize())
    if frame_queue.qsize() == 10:
        for i in range(len(frame_queue.queue[0])):
            std += [np.std([frame_queue.queue[0][i], frame_queue.queue[9][i]])]

    print(std)

# Create a window and pass it to the PoseEstimationApp class
root = tk.Tk()
app = PoseEstimationApp(root, "Pose Estimation PiP")
root.geometry("800x600")  # Set an initial window size
root.mainloop()
